/**
 * Copyright 2010 Västra Götalandsregionen
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of version 2.1 of the GNU Lesser General Public
 *   License as published by the Free Software Foundation.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 *   Boston, MA 02111-1307  USA
 */

package se.vgregion.activation.rest;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import se.vgregion.account.services.StructureService;

import javax.ws.rs.GET;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response;
import java.util.Collection;

/**
 * Service that exposes the methods of <code>StructureService</code> in a RESTful way.
 */
@Path("/")
@Produces("application/json")
public class ExternalOrgRestService {

    private StructureService structureService;

    /**
     * Constructor.
     *
     * @param structureService structureService
     */
    @Autowired
    public ExternalOrgRestService(StructureService structureService) {
        this.structureService = structureService;
    }

    /**
     * Searches for external organisation structures by query with GET request.
     * @param query query
     * @return Collection of structures in JSON format.
     */
    @GET
    @Path("/search")
    public Collection<String> search(@QueryParam("query") String query) {
        try {
            return structureService.search(query);
        } catch (Exception e) {
            throw new WebApplicationException(e, Response.Status.INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Saves a given structure.
     * @param structure E.g. "company/division/subdivision"
     */
    @PUT
    @Path("/save")
    public void save(@QueryParam("organization") String structure) {
        try {
            if (StringUtils.isBlank(structure)) {
                throw new Exception("Nothing to save");
            }

            structureService.storeExternStructurePersonDn(structure);
        } catch (Exception e) {
            throw new WebApplicationException(e, Response.Status.INTERNAL_SERVER_ERROR);
        }
    }
}
