/*
 * Decompiled with CFR 0.152.
 */
package eu.stratosphere.nephele.profiling.impl;

import eu.stratosphere.nephele.instance.InstanceConnectionInfo;
import eu.stratosphere.nephele.profiling.ProfilingException;
import eu.stratosphere.nephele.profiling.impl.types.InternalInstanceProfilingData;
import eu.stratosphere.util.StringUtils;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class InstanceProfiler {
    static final String PROC_MEMINFO = "/proc/meminfo";
    static final String PROC_STAT = "/proc/stat";
    static final String PROC_NET_DEV = "/proc/net/dev";
    private static final String LOOPBACK_INTERFACE_NAME = "lo";
    private static final Pattern CPU_PATTERN = Pattern.compile("^cpu\\s+(\\d+)\\s+(\\d+)\\s+(\\d+)\\s+(\\d+)\\s+(\\d+)\\s+(\\d+)\\s+(\\d+).+$");
    private static final Pattern NETWORK_PATTERN = Pattern.compile("^\\s*(\\w+):\\s*(\\d+)\\s+\\d+\\s+\\d+\\s+\\d+\\s+\\d+\\s+\\d+\\s+\\d+\\s+\\d+\\s+(\\d+).+$");
    private static final Pattern MEMORY_PATTERN = Pattern.compile("^\\w+:\\s*(\\d+)\\s+kB$");
    private static final int PERCENT = 100;
    private final InstanceConnectionInfo instanceConnectionInfo;
    private long lastTimestamp = 0L;
    private long lastCpuUser = 0L;
    private long lastCpuNice = 0L;
    private long lastCpuSys = 0L;
    private long lastCpuIdle = 0L;
    private long lastCpuIOWait = 0L;
    private long lastCpuIrq = 0L;
    private long lastCpuSoftirq = 0L;
    private long lastReceivedBytes = 0L;
    private long lastTramsmittedBytes = 0L;
    private long firstTimestamp;

    public InstanceProfiler(InstanceConnectionInfo instanceConnectionInfo) throws ProfilingException {
        this.instanceConnectionInfo = instanceConnectionInfo;
        this.firstTimestamp = System.currentTimeMillis();
        this.generateProfilingData(this.firstTimestamp);
    }

    InternalInstanceProfilingData generateProfilingData(long timestamp) throws ProfilingException {
        long profilingInterval = timestamp - this.lastTimestamp;
        InternalInstanceProfilingData profilingData = new InternalInstanceProfilingData(this.instanceConnectionInfo, (int)profilingInterval);
        this.updateCPUUtilization(profilingData);
        this.updateMemoryUtilization(profilingData);
        this.updateNetworkUtilization(profilingData);
        this.lastTimestamp = timestamp;
        return profilingData;
    }

    private void updateMemoryUtilization(InternalInstanceProfilingData profilingData) throws ProfilingException {
        BufferedReader in = null;
        try {
            String output;
            in = new BufferedReader(new FileReader(PROC_MEMINFO));
            long freeMemory = 0L;
            long totalMemory = 0L;
            long bufferedMemory = 0L;
            long cachedMemory = 0L;
            long cachedSwapMemory = 0L;
            int count = 0;
            while ((output = in.readLine()) != null) {
                switch (count) {
                    case 0: {
                        totalMemory = this.extractMemoryValue(output);
                        break;
                    }
                    case 1: {
                        freeMemory = this.extractMemoryValue(output);
                        break;
                    }
                    case 2: {
                        bufferedMemory = this.extractMemoryValue(output);
                        break;
                    }
                    case 3: {
                        cachedMemory = this.extractMemoryValue(output);
                        break;
                    }
                    case 4: {
                        cachedSwapMemory = this.extractMemoryValue(output);
                        break;
                    }
                }
                ++count;
            }
            profilingData.setTotalMemory(totalMemory);
            profilingData.setFreeMemory(freeMemory);
            profilingData.setBufferedMemory(bufferedMemory);
            profilingData.setCachedMemory(cachedMemory);
            profilingData.setCachedSwapMemory(cachedSwapMemory);
        }
        catch (IOException ioe) {
            throw new ProfilingException("Error while reading network utilization: " + StringUtils.stringifyException((Throwable)ioe));
        }
        finally {
            if (in != null) {
                try {
                    in.close();
                }
                catch (IOException e) {}
            }
        }
    }

    private long extractMemoryValue(String line) throws ProfilingException {
        Matcher matcher = MEMORY_PATTERN.matcher(line);
        if (!matcher.matches()) {
            throw new ProfilingException("Cannot extract memory data for profiling from line " + line);
        }
        return Long.parseLong(matcher.group(1));
    }

    private void updateNetworkUtilization(InternalInstanceProfilingData profilingData) throws ProfilingException {
        BufferedReader in = null;
        try {
            String output;
            in = new BufferedReader(new FileReader(PROC_NET_DEV));
            long receivedSum = 0L;
            long transmittedSum = 0L;
            while ((output = in.readLine()) != null) {
                Matcher networkMatcher = NETWORK_PATTERN.matcher(output);
                if (!networkMatcher.matches() || LOOPBACK_INTERFACE_NAME.equals(networkMatcher.group(1))) continue;
                receivedSum += Long.parseLong(networkMatcher.group(2));
                transmittedSum += Long.parseLong(networkMatcher.group(3));
            }
            in.close();
            in = null;
            profilingData.setReceivedBytes(receivedSum - this.lastReceivedBytes);
            profilingData.setTransmittedBytes(transmittedSum - this.lastTramsmittedBytes);
            this.lastReceivedBytes = receivedSum;
            this.lastTramsmittedBytes = transmittedSum;
        }
        catch (IOException ioe) {
            throw new ProfilingException("Error while reading network utilization: " + StringUtils.stringifyException((Throwable)ioe));
        }
        catch (NumberFormatException nfe) {
            throw new ProfilingException("Error while reading network utilization: " + StringUtils.stringifyException((Throwable)nfe));
        }
        finally {
            if (in != null) {
                try {
                    in.close();
                }
                catch (IOException e) {}
            }
        }
    }

    private void updateCPUUtilization(InternalInstanceProfilingData profilingData) throws ProfilingException {
        BufferedReader in = null;
        try {
            long cpuSoftirq;
            long deltaCpuSoftirq;
            long cpuIrq;
            long deltaCpuIrq;
            long cpuIOWait;
            long deltaCpuIOWait;
            long cpuIdle;
            long deltaCpuIdle;
            long cpuSys;
            long deltaCpuSys;
            long cpuNice;
            long deltaCpuNice;
            in = new BufferedReader(new FileReader(PROC_STAT));
            String output = in.readLine();
            if (output == null) {
                throw new ProfilingException("Cannot read CPU utilization, return value is null");
            }
            in.close();
            in = null;
            Matcher cpuMatcher = CPU_PATTERN.matcher(output);
            if (!cpuMatcher.matches()) {
                throw new ProfilingException("Cannot extract CPU utilization from output \"" + output + "\"");
            }
            long cpuUser = Long.parseLong(cpuMatcher.group(1));
            long deltaCpuUser = cpuUser - this.lastCpuUser;
            long deltaSum = deltaCpuUser + (deltaCpuNice = (cpuNice = Long.parseLong(cpuMatcher.group(2))) - this.lastCpuNice) + (deltaCpuSys = (cpuSys = Long.parseLong(cpuMatcher.group(3))) - this.lastCpuSys) + (deltaCpuIdle = (cpuIdle = Long.parseLong(cpuMatcher.group(4))) - this.lastCpuIdle) + (deltaCpuIOWait = (cpuIOWait = Long.parseLong(cpuMatcher.group(5))) - this.lastCpuIOWait) + (deltaCpuIrq = (cpuIrq = Long.parseLong(cpuMatcher.group(6))) - this.lastCpuIrq) + (deltaCpuSoftirq = (cpuSoftirq = Long.parseLong(cpuMatcher.group(7))) - this.lastCpuSoftirq);
            if (deltaSum > 0L) {
                profilingData.setIdleCPU((int)(deltaCpuIdle * 100L / deltaSum));
                profilingData.setUserCPU((int)(deltaCpuUser * 100L / deltaSum));
                profilingData.setSystemCPU((int)(deltaCpuSys * 100L / deltaSum));
                profilingData.setIoWaitCPU((int)(deltaCpuIOWait * 100L / deltaSum));
                profilingData.setHardIrqCPU((int)(deltaCpuIrq * 100L / deltaSum));
                profilingData.setSoftIrqCPU((int)(deltaCpuSoftirq * 100L / deltaSum));
            } else {
                profilingData.setIdleCPU((int)deltaCpuIdle);
                profilingData.setUserCPU((int)deltaCpuUser);
                profilingData.setSystemCPU((int)deltaCpuSys);
                profilingData.setIoWaitCPU((int)deltaCpuIOWait);
                profilingData.setHardIrqCPU((int)deltaCpuIrq);
                profilingData.setSoftIrqCPU((int)deltaCpuSoftirq);
            }
            this.lastCpuUser = cpuUser;
            this.lastCpuNice = cpuNice;
            this.lastCpuSys = cpuSys;
            this.lastCpuIdle = cpuIdle;
            this.lastCpuIOWait = cpuIOWait;
            this.lastCpuIrq = cpuIrq;
            this.lastCpuSoftirq = cpuSoftirq;
        }
        catch (IOException ioe) {
            throw new ProfilingException("Error while reading CPU utilization: " + StringUtils.stringifyException((Throwable)ioe));
        }
        catch (NumberFormatException nfe) {
            throw new ProfilingException("Error while reading CPU utilization: " + StringUtils.stringifyException((Throwable)nfe));
        }
        finally {
            if (in != null) {
                try {
                    in.close();
                }
                catch (IOException e) {}
            }
        }
    }

    public InternalInstanceProfilingData generateCheckpointProfilingData() throws ProfilingException {
        long profilingInterval = System.currentTimeMillis() - this.firstTimestamp;
        InternalInstanceProfilingData profilingData = new InternalInstanceProfilingData(this.instanceConnectionInfo, (int)profilingInterval);
        this.updateCPUUtilization(profilingData);
        this.updateMemoryUtilization(profilingData);
        this.updateNetworkUtilization(profilingData);
        return profilingData;
    }
}

